package argon2

import (
	"bytes"
	"testing"
)

// Repeat returns a slice containing n copies of v.
func repeat(v uint8, n int) []byte {
	b := make([]byte, n)
	for i := range b {
		b[i] = v
	}
	return b
}

// Runs argon2 with logging enabled, for debugging purposes
func TestDebug(t *testing.T) {
	var out [8]uint8
	argon2(out[:], repeat(0, 16), repeat(1, 8), nil, nil, 1, 8, 3, t)
}

// Runs the test vector from the official repository
func TestArgon_Vector(t *testing.T) {
	msg := repeat(0x1, 32)
	salt := repeat(0x2, 16)
	key := repeat(0x3, 8)
	data := repeat(0x4, 12)
	want := []byte{0x51, 0x2b, 0x39, 0x1b, 0x6f, 0x11, 0x62, 0x97, 0x53, 0x71, 0xd3, 0x09, 0x19, 0x73, 0x42, 0x94, 0xf8, 0x68, 0xe3, 0xbe, 0x39, 0x84, 0xf3, 0xc1, 0xa1, 0x3a, 0x4d, 0xb9, 0xfa, 0xbe, 0x4a, 0xcb}
	out := make([]byte, len(want))
	argon2(out, msg, salt, key, data, 4, 32, 3, t)
	if !bytes.Equal(want, out) {
		t.Errorf("got % x, want % x\n", out, want)
	}
}

func TestArgon(t *testing.T) {
	var tests = []struct {
		n    uint32
		mem  uint32
		par  uint32
		want []byte
	}{
		{n: 3, mem: 8, par: 1, want: []byte{0x87, 0x22, 0x4f, 0xd2, 0x26, 0xbb, 0xf0, 0x76}},
		{n: 4, mem: 8, par: 1, want: []byte{0x65, 0x5f, 0x33, 0x24, 0xe7, 0x29, 0xa9, 0x3a}},
		{n: 5, mem: 8, par: 1, want: []byte{0x56, 0x61, 0x52, 0xd4, 0x5a, 0xd2, 0x5a, 0x47}},
		{n: 6, mem: 8, par: 1, want: []byte{0x80, 0x99, 0x38, 0x39, 0xd9, 0x19, 0x20, 0x5a}},
		{n: 7, mem: 8, par: 1, want: []byte{0x57, 0x33, 0x09, 0x7e, 0xf4, 0x9b, 0x19, 0xa0}},
		{n: 8, mem: 8, par: 1, want: []byte{0xff, 0x7d, 0x02, 0x0a, 0x8d, 0x25, 0xfa, 0x87}},
		{n: 9, mem: 8, par: 1, want: []byte{0x95, 0xae, 0xef, 0x81, 0xd9, 0x15, 0x61, 0x06}},
		{n: 10, mem: 8, par: 1, want: []byte{0x40, 0xfb, 0xe9, 0xd3, 0xa6, 0x50, 0xb7, 0xcb}},

		{n: 3, mem: 16, par: 1, want: []byte{0x44, 0x82, 0x67, 0x1c, 0x29, 0x5e, 0x5b, 0x84}},
		{n: 3, mem: 32, par: 1, want: []byte{0xc3, 0xb2, 0x0a, 0xdb, 0x9f, 0x6b, 0xc0, 0xed}},
		{n: 3, mem: 64, par: 1, want: []byte{0x6b, 0x49, 0x39, 0xf5, 0x15, 0x42, 0x5d, 0xe3}},
		{n: 3, mem: 128, par: 1, want: []byte{0x1e, 0x50, 0xad, 0x31, 0x02, 0x7d, 0xa5, 0xbc}},
		{n: 3, mem: 256, par: 1, want: []byte{0x49, 0x83, 0x61, 0xdb, 0x18, 0xab, 0xaf, 0xb8}},
		{n: 3, mem: 512, par: 1, want: []byte{0xd0, 0x48, 0x7d, 0x84, 0xcb, 0xd3, 0x9f, 0x10}},
		{n: 3, mem: 1024, par: 1, want: []byte{0x51, 0xf7, 0xb0, 0xdb, 0x3b, 0xdb, 0xef, 0xb8}},

		{n: 3, mem: 16, par: 2, want: []byte{0x73, 0x16, 0xaf, 0x5d, 0x31, 0xe9, 0xed, 0xea}},
		{n: 3, mem: 32, par: 4, want: []byte{0x16, 0xb7, 0x3b, 0xb9, 0x69, 0xcd, 0x6c, 0x63}},
		{n: 3, mem: 64, par: 8, want: []byte{0x25, 0xf3, 0x85, 0x08, 0x60, 0x5d, 0x35, 0x45}},
		{n: 3, mem: 128, par: 16, want: []byte{0xe3, 0xd3, 0xe3, 0x72, 0xdf, 0x0a, 0x22, 0x7b}},
		{n: 3, mem: 256, par: 32, want: []byte{0x5c, 0xf9, 0x79, 0x0a, 0x9c, 0xc1, 0x05, 0x6b}},
		{n: 3, mem: 512, par: 64, want: []byte{0xa6, 0xc9, 0x71, 0xcc, 0x99, 0x4d, 0xcf, 0x8f}},

		{n: 3, mem: 8, par: 1, want: []byte{0x3c, 0x9a, 0x65, 0x86}},
		{n: 3, mem: 8, par: 1, want: []byte{0x20, 0x4d, 0x2e, 0x37, 0x4a}},
		{n: 3, mem: 8, par: 1, want: []byte{0xc9, 0xd5, 0xd9, 0x75, 0x36, 0x89}},
		{n: 3, mem: 8, par: 1, want: []byte{0x8b, 0x32, 0x11, 0x8e, 0xf6, 0xf6, 0x3d}},
		{n: 3, mem: 8, par: 1, want: []byte{0x87, 0x22, 0x4f, 0xd2, 0x26, 0xbb, 0xf0, 0x76}},
		{n: 3, mem: 8, par: 1, want: []byte{0x05, 0xe6, 0xbe, 0x6c, 0x27, 0x1e, 0x93, 0xac, 0x74}},
		{n: 3, mem: 8, par: 1, want: []byte{0xee, 0xb4, 0x11, 0xb4, 0xfa, 0x1a, 0x6c, 0xe8, 0x14, 0x2b}},
		{n: 3, mem: 8, par: 1, want: []byte{0x1e, 0x23, 0xa3, 0x87, 0x53, 0x8e, 0x21, 0xbf, 0x07, 0x0c, 0x59}},
		{n: 3, mem: 8, par: 1, want: []byte{0x7a, 0x0c, 0x41, 0x54, 0x5e, 0x20, 0x38, 0x23, 0x57, 0x16, 0x61, 0xcf}},
		{n: 3, mem: 8, par: 1, want: []byte{0x1c, 0xe5, 0xcf, 0xee, 0x7f, 0x6f, 0x43, 0x55, 0x67, 0x14, 0xaa, 0x5b, 0x8e}},
		{n: 3, mem: 8, par: 1, want: []byte{0xef, 0x92, 0x6c, 0xa6, 0x6c, 0x25, 0x74, 0x3c, 0xa3, 0x32, 0xfa, 0x0a, 0xdd, 0x2b}},
		{n: 3, mem: 8, par: 1, want: []byte{0x47, 0xcd, 0xb0, 0xd0, 0x10, 0x88, 0x54, 0x76, 0xb0, 0x0c, 0x30, 0x1d, 0x05, 0x59, 0xc3}},
		{n: 3, mem: 8, par: 1, want: []byte{0x91, 0x69, 0xdb, 0x87, 0xc4, 0xd1, 0xc7, 0x80, 0xa1, 0xdb, 0x8a, 0x47, 0x5d, 0xc0, 0xac, 0xcd}},

		{n: 3, mem: 8, par: 1, want: []byte{0x06, 0x4f, 0xc0, 0x84, 0x30, 0x11, 0xab, 0x42, 0xe8, 0x74, 0xd2, 0xba, 0x3c, 0xeb, 0x26, 0x41, 0xb0, 0x50, 0x88, 0xc1, 0x3a, 0x13, 0xdf, 0x5c, 0x6f, 0xf1, 0xf3, 0xf3, 0xc2, 0x61, 0xf6, 0xd1}},
		{n: 3, mem: 8, par: 1, want: []byte{0x4b, 0x63, 0xb4, 0x2a, 0x51, 0x2b, 0x81, 0x5e, 0x75, 0x0d, 0xed, 0x7a, 0x87, 0x14, 0x42, 0x1c, 0xcd, 0x59, 0x85, 0x66, 0xe4, 0xbb, 0x5a, 0xf5, 0xaf, 0xc8, 0xb3, 0x7c, 0x58, 0x42, 0xfc, 0x14, 0x0e}},
		{n: 3, mem: 8, par: 1, want: []byte{0xaf, 0xed, 0xcf, 0x2d, 0xbf, 0x00, 0xf0, 0xad, 0x5c, 0x18, 0x4e, 0x82, 0xbc, 0xc3, 0x10, 0x85, 0xed, 0x9c, 0x63, 0x54, 0x43, 0xe9, 0xc9, 0x88, 0x71, 0xe4, 0xfa, 0x9a, 0x6a, 0x54, 0x15, 0xb5, 0xc0, 0x07, 0xdd, 0xaf, 0xac, 0x48, 0x71, 0xf0, 0xab, 0xe0, 0xa6, 0xdb, 0x35, 0x52, 0xbf, 0x3a, 0x07, 0x80, 0x78, 0xd2, 0x7d, 0xa2, 0x41, 0x52, 0x00, 0x6f, 0xfe, 0xd0, 0x8b, 0x17, 0xe8, 0xe9}},
		{n: 3, mem: 8, par: 1, want: []byte{0x7a, 0xf9, 0x5b, 0x8e, 0x44, 0x63, 0x48, 0x67, 0x15, 0x87, 0x4c, 0x35, 0xf6, 0x7f, 0x52, 0xd1, 0xe7, 0x97, 0x95, 0x12, 0xce, 0x49, 0x5c, 0x06, 0xf9, 0xd4, 0xf8, 0xda, 0xc2, 0xc4, 0x61, 0xa1, 0xfe, 0x0a, 0x1f, 0xd5, 0x26, 0x89, 0x21, 0xd9, 0xdb, 0x03, 0x15, 0x6f, 0xa2, 0xf6, 0x83, 0x7d, 0x7a, 0xf6, 0xde, 0x3d, 0xd7, 0x8b, 0x9d, 0x03, 0x7f, 0x6c, 0xd9, 0xe0, 0x99, 0x1e, 0x52, 0x82, 0xbb}},

		{n: 3, mem: 8, par: 1, want: []byte{0x7a, 0x3a, 0x97, 0xa3, 0x65, 0xcd, 0x6e, 0xb3, 0x64, 0x32, 0xbc, 0xee, 0x38, 0xa5, 0x32, 0xef, 0x85, 0x2d, 0x30, 0x7f, 0x65, 0x56, 0x0b, 0xb3, 0xfb, 0x8c, 0x94, 0x60, 0x7b, 0xbc, 0xc4, 0x52, 0xb0, 0x4a, 0x26, 0x99, 0xaf, 0x40, 0x7f, 0x25, 0x62, 0x09, 0xc3, 0xe6, 0xd9, 0x8a, 0xcd, 0xac, 0xdc, 0x9c, 0xc2, 0x25, 0xd2, 0x2a, 0x96, 0x9f, 0x80, 0xb6, 0x5a, 0x3d, 0xc5, 0x10, 0xa0, 0x03, 0xe5, 0x32, 0xae, 0x17, 0x3b, 0x76, 0x8b, 0x91, 0x4d, 0x48, 0xc8, 0x42, 0x43, 0xee, 0x23, 0x65, 0xb7, 0xd9, 0xae, 0x4e, 0x29, 0xf4, 0x6a, 0xcd, 0x19, 0x4e, 0x5b, 0x13, 0xa4, 0x38, 0xf3, 0x57}},
		{n: 3, mem: 8, par: 1, want: []byte{0x0a, 0x25, 0xa1, 0x50, 0x63, 0xfe, 0x10, 0x36, 0x7c, 0x46, 0x4d, 0xab, 0x8d, 0xb7, 0x30, 0xfd, 0x46, 0x31, 0xee, 0x3b, 0x6d, 0xf3, 0xdb, 0xef, 0x8a, 0xcd, 0xbf, 0x9d, 0x9a, 0x4f, 0x61, 0x74, 0x26, 0x35, 0xae, 0x39, 0x70, 0xdb, 0xe9, 0x73, 0xf7, 0xb6, 0x62, 0x04, 0x83, 0x6e, 0x90, 0x9d, 0x5d, 0x91, 0x07, 0x15, 0x7b, 0xa4, 0x60, 0x23, 0x16, 0x06, 0x15, 0x7c, 0xd2, 0x5d, 0x45, 0x0b, 0x89, 0xf7, 0xd1, 0x8c, 0xdc, 0x48, 0xe2, 0x9e, 0x54, 0xe0, 0x65, 0x99, 0xc4, 0xf7, 0x5b, 0xed, 0x9b, 0x9f, 0x2e, 0xd9, 0x78, 0x23, 0x5e, 0xc0, 0x8d, 0x4f, 0x91, 0x29, 0x0c, 0xcb, 0x1d, 0x8f, 0x85, 0x43, 0xd3, 0x3b, 0xbe, 0x5b, 0x85, 0x33, 0x3e, 0xaa, 0xca, 0xc3, 0x83, 0x23, 0x6e, 0x4f, 0x26, 0x28, 0x77, 0xf3, 0xf4, 0xb4, 0x87, 0x66, 0xc4, 0x84, 0xd8, 0xbe, 0xba, 0x5a, 0x5f, 0xba}},
	}

	msg := repeat(0x0, 16)
	salt := repeat(0x1, 8)

	for _, tt := range tests {
		out := make([]byte, len(tt.want))
		argon2(out, msg, salt, nil, nil, tt.par, tt.mem, tt.n, nil)
		if !bytes.Equal(out, tt.want) {
			t.Errorf("n=%d, mem=%d, par=%d, len=%d: got % x, want % x\n", tt.n, tt.mem, tt.par, len(tt.want), out, tt.want)
		}
	}
}

func TestAllocs(t *testing.T) {
	pw := repeat(0x0, 16)
	salt := repeat(0x1, 8)
	out := make([]byte, 32)
	allocs := testing.AllocsPerRun(100, func() {
		argon2(out, pw, salt, nil, nil, 4, 32, 3, nil)
	})
	if allocs > 6 {
		t.Errorf("%v allocs, want <=6", allocs)
	}
}

func benchArgon(b *testing.B, par uint8, mem, n uint32) {
	msg := repeat(0x0, 16)
	salt := repeat(0x1, 8)
	out := make([]byte, 8)
	b.SetBytes(int64(mem) << 10)
	for i := 0; i < b.N; i++ {
		argon2(out, msg, salt, nil, nil, uint32(par), mem, n, nil)
	}
}

func BenchmarkArgon8KiB(b *testing.B)   { benchArgon(b, 1, 8, 3) }
func BenchmarkArgon80KiB(b *testing.B)  { benchArgon(b, 1, 80, 3) }
func BenchmarkArgon800KiB(b *testing.B) { benchArgon(b, 1, 800, 3) }

func BenchmarkArgon3N(b *testing.B)   { benchArgon(b, 1, 8, 3) }
func BenchmarkArgon10N(b *testing.B)  { benchArgon(b, 1, 8, 10) }
func BenchmarkArgon100N(b *testing.B) { benchArgon(b, 1, 8, 100) }

func BenchmarkArgon1P(b *testing.B) { benchArgon(b, 1, 64, 3) }
func BenchmarkArgon2P(b *testing.B) { benchArgon(b, 2, 64, 3) }
func BenchmarkArgon4P(b *testing.B) { benchArgon(b, 4, 64, 3) }

//func BenchmarkArgon_4MiB(b *testing.B) { benchArgon(b, 1, 4096, 3) }
