package argon2

import (
	"bytes"
	"testing"
)

// Repeat returns a slice containing n copies of v.
func repeat(v uint8, n int) []byte {
	b := make([]byte, n)
	for i := range b {
		b[i] = v
	}
	return b
}

// Runs argon2 with logging enabled, for debugging purposes
func TestDebug(t *testing.T) {
	var out [8]uint8
	argon2(out[:], repeat(0, 16), repeat(1, 8), nil, nil, 1, 8, 3, t)
}

// Runs the test vector from the official repository
func TestArgon_Vector(t *testing.T) {
	msg := repeat(0x1, 32)
	salt := repeat(0x2, 16)
	key := repeat(0x3, 8)
	data := repeat(0x4, 12)
	want := []byte{0x57, 0xb0, 0x61, 0x3b, 0xfd, 0xd4, 0x13, 0x1a, 0x0c, 0x34, 0x88, 0x34, 0xc6, 0x72, 0x9c, 0x2c, 0x72, 0x29, 0x92, 0x1e, 0x6b, 0xba, 0x37, 0x66, 0x5d, 0x97, 0x8c, 0x4f, 0xe7, 0x17, 0x5e, 0xd2}
	out := make([]byte, len(want))
	argon2(out, msg, salt, key, data, 4, 16, 3, t)
	if !bytes.Equal(want, out) {
		t.Errorf("got % x, want % x\n", out, want)
	}
}

func TestArgon(t *testing.T) {
	var tests = []struct {
		n    uint32
		mem  uint32
		par  uint32
		want []byte
	}{
		{n: 3, mem: 8, par: 1, want: []byte{0x01, 0x99, 0x83, 0x87, 0x43, 0xcd, 0xe9, 0x08}},
		{n: 4, mem: 8, par: 1, want: []byte{0x6b, 0xec, 0xaf, 0x23, 0xe2, 0xfb, 0x1f, 0x95}},
		{n: 5, mem: 8, par: 1, want: []byte{0x89, 0x0f, 0xa4, 0x3c, 0xcf, 0x71, 0xcd, 0xf0}},
		{n: 6, mem: 8, par: 1, want: []byte{0x58, 0x1f, 0x5d, 0xf0, 0x9e, 0xb2, 0x7e, 0xa0}},
		{n: 7, mem: 8, par: 1, want: []byte{0x80, 0x52, 0xf4, 0x2a, 0xd3, 0xbd, 0x92, 0x4e}},
		{n: 8, mem: 8, par: 1, want: []byte{0x4b, 0xe6, 0x59, 0xc5, 0x6f, 0xf0, 0xfd, 0xe7}},
		{n: 9, mem: 8, par: 1, want: []byte{0xb1, 0xab, 0xe5, 0xef, 0xe7, 0x6c, 0xc9, 0x1e}},
		{n: 10, mem: 8, par: 1, want: []byte{0x32, 0xb7, 0x41, 0x65, 0xb1, 0x99, 0x1d, 0xc4}},

		{n: 3, mem: 16, par: 1, want: []byte{0x86, 0xb1, 0xa4, 0x76, 0x4c, 0x36, 0xa6, 0xba}},
		{n: 3, mem: 32, par: 1, want: []byte{0xeb, 0x56, 0x5a, 0x14, 0xe5, 0x36, 0x66, 0x87}},
		{n: 3, mem: 64, par: 1, want: []byte{0x60, 0x11, 0xa6, 0x39, 0x32, 0x37, 0xc0, 0xd3}},
		{n: 3, mem: 128, par: 1, want: []byte{0x3b, 0x94, 0x37, 0xbd, 0x2e, 0x34, 0x6b, 0xda}},
		{n: 3, mem: 256, par: 1, want: []byte{0x0a, 0xce, 0x73, 0xce, 0x74, 0x58, 0x77, 0xa4}},
		{n: 3, mem: 512, par: 1, want: []byte{0xe1, 0xe6, 0x8b, 0x4d, 0x9e, 0xfc, 0x08, 0x20}},
		{n: 3, mem: 1024, par: 1, want: []byte{0xdc, 0x43, 0xf1, 0xf5, 0x5a, 0x63, 0x6c, 0xbf}},

		{n: 3, mem: 16, par: 2, want: []byte{0xc4, 0xa5, 0xdd, 0x0b, 0xe8, 0x8e, 0x55, 0x73}},
		{n: 3, mem: 32, par: 4, want: []byte{0xe9, 0xd5, 0xff, 0x75, 0xce, 0x66, 0xa2, 0x23}},
		{n: 3, mem: 64, par: 8, want: []byte{0x30, 0x4d, 0xea, 0x04, 0xe1, 0x59, 0xae, 0x69}},
		{n: 3, mem: 128, par: 16, want: []byte{0x3d, 0x78, 0x1e, 0xd8, 0xd3, 0x91, 0xfa, 0x87}},
		{n: 3, mem: 256, par: 32, want: []byte{0xb4, 0x37, 0xa7, 0x76, 0xbf, 0xd5, 0x36, 0x9f}},
		{n: 3, mem: 512, par: 64, want: []byte{0xc4, 0xd6, 0x86, 0xce, 0x05, 0x61, 0x2c, 0x05}},

		{n: 3, mem: 8, par: 1, want: []byte{0x3d, 0x91, 0x49, 0xc8}},
		{n: 3, mem: 8, par: 1, want: []byte{0x48, 0x1c, 0xc7, 0x1b, 0xcc}},
		{n: 3, mem: 8, par: 1, want: []byte{0xc2, 0x4d, 0xf2, 0x1f, 0x0e, 0x77}},
		{n: 3, mem: 8, par: 1, want: []byte{0x0c, 0xe9, 0xa8, 0xf8, 0x51, 0x72, 0x14}},
		{n: 3, mem: 8, par: 1, want: []byte{0x01, 0x99, 0x83, 0x87, 0x43, 0xcd, 0xe9, 0x08}},
		{n: 3, mem: 8, par: 1, want: []byte{0x2a, 0x8c, 0x92, 0x25, 0xc0, 0xdd, 0x71, 0xb8, 0x15}},
		{n: 3, mem: 8, par: 1, want: []byte{0x86, 0x1a, 0x76, 0x5e, 0xfa, 0x4f, 0x92, 0x64, 0x38, 0x47}},
		{n: 3, mem: 8, par: 1, want: []byte{0x9a, 0xbf, 0x93, 0x93, 0x09, 0xaf, 0xce, 0xd8, 0x33, 0x08, 0xdb}},
		{n: 3, mem: 8, par: 1, want: []byte{0x12, 0xec, 0xa7, 0x0d, 0x22, 0x0c, 0xde, 0x4e, 0x34, 0xe3, 0x2f, 0x7c}},
		{n: 3, mem: 8, par: 1, want: []byte{0xd1, 0xc3, 0x96, 0x24, 0x62, 0x4e, 0x00, 0xf7, 0x54, 0x8e, 0xbe, 0xef, 0xec}},
		{n: 3, mem: 8, par: 1, want: []byte{0x81, 0xce, 0x2d, 0xa8, 0xa8, 0x8c, 0x06, 0xcb, 0x50, 0xf8, 0x23, 0x55, 0x90, 0x16}},
		{n: 3, mem: 8, par: 1, want: []byte{0x5d, 0x03, 0xbd, 0x23, 0xec, 0x11, 0x42, 0xd0, 0x9a, 0xea, 0x16, 0xef, 0x6a, 0x3e, 0xea}},
		{n: 3, mem: 8, par: 1, want: []byte{0x34, 0xd8, 0x38, 0x7f, 0xa4, 0xe0, 0xb4, 0x85, 0x95, 0x1a, 0xaf, 0x07, 0x7d, 0x90, 0xce, 0x8b}},

		{n: 3, mem: 8, par: 1, want: []byte{0xdd, 0x21, 0xf8, 0x2c, 0xed, 0xda, 0x82, 0xda, 0x0f, 0xf7, 0xea, 0xf4, 0x1f, 0x13, 0x84, 0x18, 0xb9, 0x4f, 0x09, 0x2a, 0x04, 0xff, 0xb4, 0x1d, 0x56, 0xfe, 0xb0, 0xcb, 0xb4, 0x19, 0xb5, 0xaf}},
		{n: 3, mem: 8, par: 1, want: []byte{0x1f, 0x96, 0x43, 0x97, 0xa3, 0x83, 0x83, 0x37, 0x5c, 0x2f, 0x1f, 0x8e, 0x38, 0x7b, 0xbc, 0x74, 0xfd, 0xc4, 0xa9, 0x32, 0x95, 0xb1, 0x4f, 0xd9, 0xcd, 0x0c, 0x09, 0xec, 0x9f, 0x10, 0x5f, 0x2c, 0x6a}},
		{n: 3, mem: 8, par: 1, want: []byte{0xd8, 0x45, 0x2f, 0x29, 0xba, 0x32, 0x87, 0xdc, 0xa7, 0x93, 0x20, 0x87, 0xdc, 0xf5, 0xa3, 0x8a, 0xb3, 0xfa, 0x83, 0x3c, 0x28, 0xd5, 0xf4, 0x67, 0x72, 0x13, 0x1c, 0x7d, 0x8d, 0xaf, 0x42, 0xf9, 0x30, 0x19, 0x2d, 0xb9, 0x3b, 0x76, 0xc7, 0xbe, 0xcf, 0x08, 0x47, 0xd9, 0xc7, 0xd3, 0x14, 0x34, 0x50, 0x5f, 0x79, 0xa2, 0xe0, 0x8a, 0x39, 0xdc, 0x6c, 0xd6, 0x8e, 0x26, 0x61, 0x76, 0xa6, 0x6b}},
		{n: 3, mem: 8, par: 1, want: []byte{0x48, 0x88, 0xd4, 0x42, 0xa7, 0x32, 0x32, 0x85, 0xd9, 0x98, 0x46, 0x29, 0x77, 0x68, 0xfd, 0x74, 0x37, 0x9e, 0xce, 0x3d, 0xa0, 0x76, 0xb7, 0x46, 0x37, 0x18, 0xf5, 0x9f, 0x38, 0xc2, 0x5f, 0x8f, 0x7c, 0x4a, 0xb2, 0x08, 0x4c, 0x24, 0xa2, 0xf4, 0x45, 0x36, 0x3e, 0x31, 0x89, 0xd0, 0xdd, 0x6d, 0xb2, 0x74, 0x5f, 0x25, 0x3b, 0xe1, 0x27, 0x79, 0x2f, 0x74, 0x52, 0x06, 0x30, 0x9d, 0x07, 0x32, 0xb5}},
		{n: 3, mem: 8, par: 1, want: []byte{0x71, 0xd3, 0xe3, 0xd9, 0xb9, 0xb7, 0xad, 0x04, 0x87, 0x54, 0x3a, 0xcd, 0xbe, 0x60, 0x35, 0x1e, 0xff, 0x3e, 0x5f, 0xee, 0xb8, 0x1a, 0x3a, 0xec, 0x01, 0x2a, 0x08, 0x3c, 0x05, 0xe5, 0xcf, 0x26, 0x8e, 0x19, 0x1a, 0x90, 0xc6, 0x94, 0x10, 0xd3, 0xf4, 0x06, 0x66, 0x82, 0x9c, 0x2b, 0x8c, 0x95, 0x92, 0x01, 0xe6, 0x8a, 0xf8, 0x6e, 0xcb, 0xee, 0x12, 0x31, 0x2c, 0xb3, 0x26, 0x9c, 0x96, 0x56, 0xa2, 0x36, 0xb2, 0xb2, 0x37, 0xb1, 0x66, 0xfe, 0x7c, 0xd5, 0x53, 0xc0, 0xf6, 0xc9, 0xc4, 0xf2, 0x7a, 0xc6, 0x35, 0x95, 0xb6, 0x4c, 0x81, 0x93, 0x52, 0x34, 0x54, 0x69, 0x91, 0xd4, 0xc2, 0x37}},
		{n: 3, mem: 8, par: 1, want: []byte{0x7a, 0x7c, 0xab, 0xc6, 0x7b, 0xbc, 0x4b, 0x11, 0xa5, 0x4c, 0xed, 0x73, 0x29, 0xbb, 0xde, 0x50, 0x97, 0x9f, 0x8c, 0x33, 0xa7, 0x3e, 0x24, 0xc7, 0xf7, 0x9a, 0xe9, 0xac, 0xbb, 0xda, 0xa7, 0x59, 0x3d, 0x16, 0x05, 0xb5, 0x1d, 0x09, 0xa1, 0x08, 0xb1, 0x5f, 0xb3, 0xa6, 0x62, 0xac, 0x96, 0xa9, 0x00, 0x4b, 0x6d, 0x49, 0xc3, 0x34, 0x4d, 0x6a, 0x77, 0x55, 0x59, 0x1f, 0xaa, 0x44, 0xee, 0x8d, 0xb1, 0x26, 0x02, 0x3b, 0xfc, 0xd8, 0x68, 0xc7, 0xcd, 0x96, 0x16, 0xf6, 0xe6, 0x79, 0x02, 0x56, 0x64, 0x7d, 0xbd, 0xa8, 0x08, 0x90, 0x7d, 0xe2, 0x6d, 0x95, 0x63, 0x2e, 0x24, 0xb4, 0x62, 0xc7, 0x7f, 0x0b, 0x0a, 0x04, 0x28, 0xd9, 0xda, 0xa3, 0x6f, 0xda, 0xda, 0x4f, 0xc3, 0x7a, 0x5b, 0xfe, 0x54, 0xd6, 0xc0, 0x4a, 0x95, 0x78, 0xce, 0xe8, 0xed, 0x82, 0x15, 0xe8, 0xca, 0x23, 0x39, 0x05}},
	}

	msg := repeat(0x0, 16)
	salt := repeat(0x1, 8)

	for _, tt := range tests {
		out := make([]byte, len(tt.want))
		argon2(out, msg, salt, nil, nil, tt.par, tt.mem, tt.n, nil)
		if !bytes.Equal(out, tt.want) {
			t.Errorf("n=%d, mem=%d, par=%d, len=%d: got % x, want % x\n", tt.n, tt.mem, tt.par, len(tt.want), out, tt.want)
		}
	}
}

func TestAllocs(t *testing.T) {
	pw := repeat(0x0, 16)
	salt := repeat(0x1, 8)
	out := make([]byte, 32)
	allocs := testing.AllocsPerRun(100, func() {
		argon2(out, pw, salt, nil, nil, 4, 32, 3, nil)
	})
	if allocs > 6 {
		t.Errorf("%v allocs, want <=6", allocs)
	}
}

func benchArgon(b *testing.B, par uint8, mem, n uint32) {
	msg := repeat(0x0, 16)
	salt := repeat(0x1, 8)
	out := make([]byte, 8)
	b.SetBytes(int64(mem) << 10)
	for i := 0; i < b.N; i++ {
		argon2(out, msg, salt, nil, nil, uint32(par), mem, n, nil)
	}
}

func BenchmarkArgon8KiB(b *testing.B)   { benchArgon(b, 1, 8, 3) }
func BenchmarkArgon80KiB(b *testing.B)  { benchArgon(b, 1, 80, 3) }
func BenchmarkArgon800KiB(b *testing.B) { benchArgon(b, 1, 800, 3) }

func BenchmarkArgon3N(b *testing.B)   { benchArgon(b, 1, 8, 3) }
func BenchmarkArgon10N(b *testing.B)  { benchArgon(b, 1, 8, 10) }
func BenchmarkArgon100N(b *testing.B) { benchArgon(b, 1, 8, 100) }

func BenchmarkArgon1P(b *testing.B) { benchArgon(b, 1, 64, 3) }
func BenchmarkArgon2P(b *testing.B) { benchArgon(b, 2, 64, 3) }
func BenchmarkArgon4P(b *testing.B) { benchArgon(b, 4, 64, 3) }

//func BenchmarkArgon_4MiB(b *testing.B) { benchArgon(b, 1, 4096, 3) }
